<?php
/*
 * @package   bfNetwork
 * @copyright Copyright (C) 2011,2012,2013,2014,2015,2016,2017,2018,2019,2020,2021,2022 Blue Flame Digital Solutions Ltd. All rights reserved.
 * @license   GNU General Public License version 3 or later
 *
 * @see       https://mySites.guru/
 * @see       https://www.phil-taylor.com/
 *
 * @author    Phil Taylor / Blue Flame Digital Solutions Limited.
 *
 * bfNetwork is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * bfNetwork is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this package.  If not, see http://www.gnu.org/licenses/
 *
 * If you have any questions regarding this code, please contact phil@phil-taylor.com
 */

// Decrypt or die
require 'bfEncrypt.php';

class bfBaseline
{
    // types of data retrieval.
    const BASELINE = 'baseline';

    const COMPARE  = 'compare';

    /**
     * We pass the command to run as a simple integer in our encrypted request this is mainly to speed up the decryption
     * process, plus its a single digit(or 2) rather than a huge string to remember :-).
     */
    private $_methods = array(
        1 => 'establish',
        2 => 'clear',
        3 => 'getChanges',
    );

    /**
     * Pointer to the Joomla Database Object..
     *
     * @var bfDb
     */
    private $db;

    /**
     * Incoming decrypted vars from the request.
     *
     * @var stdClass
     */
    private $_dataObj;

    /**
     * @param stdClass $dataObj
     */
    public function __construct($dataObj)
    {
        // init Joomla
        require 'bfInitWordpress.php';
        require_once 'bfActivitylog.php';

        // Set the request vars
        $this->_dataObj = $dataObj;

        // require all we need to access WordPress API
        require dirname(__FILE__) . '/bfDb.php';

        $this->db = new bfDb();
    }

    /**
     * I'm the controller - I run methods based on the request integer.
     */
    public function run()
    {
        if (property_exists($this->_dataObj, 'c')) {
            $c = (int) $this->_dataObj->c;
            if (array_key_exists($c, $this->_methods)) {
                bfLog::log('Calling methd ' . $this->_methods[$c]);
                // call the right method
                $this->{$this->_methods[$c]} ();
            } else {
                // Die if an unknown function
                bfEncrypt::reply('error', 'No Such method #err1 - ' . $c);
            }
        } else {
            // Die if an unknown function
            bfEncrypt::reply('error', 'No Such method #err2');
        }
    }

    public function establish()
    {
        bfLog::log('Creating our baseline database tables');

        $this->db->setQuery('SHOW TABLES LIKE "bf_baseline"');
        if ($hasbfBaseline = $this->db->loadResult()) {
            $this->db->setQuery('DROP TABLE IF EXISTS `bf_baseline`');
            $this->db->query();
        }

        $this->db->setQuery('SHOW TABLES LIKE "bf_files"');
        if ($hasbfFiles = $this->db->loadResult()) {
            $this->db->setQuery('CREATE TABLE bf_baseline LIKE bf_files');
            $this->db->query();
            $this->db->setQuery('INSERT INTO bf_baseline SELECT * FROM bf_files');
            $this->db->query();
        }

        bfEncrypt::reply(
            bfEncrypt::SUCCESS,
            array(
                'establish'     => 'done',
                'hasbfFiles'    => $hasbfFiles,
                'hasbfBaseline' => $hasbfBaseline,
            )
        );
    }

    public function clear()
    {
        bfLog::log('Clearing our baseline database tables');

        $this->db->setQuery('SHOW TABLES LIKE "bf_baseline"');
        if ($this->db->loadResult()) {
            $this->db->setQuery('DROP TABLE IF EXISTS `bf_baseline`');
            $this->db->query();
        }

        bfEncrypt::reply(bfEncrypt::SUCCESS, array(
            'clear' => 'done',
        ));
    }

    public function getChanges()
    {
        bfLog::log('Getting our baseline changes');

        $dataSQL = $this->getDataSQL();

        $totalSQL = str_replace(' * ', ' count(*) ', $dataSQL);

        $offset = isset($this->_dataObj->ls) ? $this->_dataObj->ls : 0;
        $count  = isset($this->_dataObj->limit) ? $this->_dataObj->limit : 500;

        $this->db->setQuery($dataSQL . sprintf(' LIMIT %s, %s', (int) $offset, (int) $count));
        $data = $this->db->loadObjectList();

        $this->db->setQuery($totalSQL);
        $total = $this->db->loadResult();

        bfEncrypt::reply(bfEncrypt::SUCCESS, array(
            'data'  => $data,
            'total' => $total,
        ));
    }

    private function getDataSQL()
    {
        $dataSQL = '';

        switch ($this->_dataObj->what) {
            case 'fileshashchanged':
                $dataSQL = ' SELECT * FROM OLDTABLE 
                            LEFT JOIN bf_files ON bf_files.filewithpath = OLDTABLE.filewithpath
                            WHERE bf_files.currenthash != OLDTABLE.currenthash ORDER BY OLDTABLE.filewithpath ASC';
                break;
            case 'filesdeleted':
                $dataSQL = 'SELECT * FROM OLDTABLE WHERE filewithpath NOT IN (SELECT filewithpath FROM bf_files)  ORDER BY OLDTABLE.filewithpath ASC';
                break;
            case 'filesadded':
                $dataSQL = 'SELECT * FROM bf_files WHERE filewithpath NOT IN (SELECT filewithpath FROM OLDTABLE) ORDER BY filemtime ASC';
                break;
            case 'filesmodified':
                $dataSQL = 'SELECT * FROM bf_files 
                            WHERE filewithpath IN (SELECT filewithpath FROM OLDTABLE) 
                            AND filemtime > ' . strtotime($this->_dataObj->lastbaseline) . ' ORDER BY filemtime ASC';
                break;
            default:
                bfEncrypt::reply(bfEncrypt::ERROR, array(
                    'what' => $this->_dataObj->what . ':NotAcceptable',
                ));
                break;
        }

        switch ($this->_dataObj->type) {
            case self::COMPARE:
                $dataSQL = str_replace('OLDTABLE', 'bf_files_last', $dataSQL);
                break;
            case self::BASELINE:
                $dataSQL = str_replace('OLDTABLE', 'bf_baseline', $dataSQL);
                break;
        }

        return $dataSQL;
    }
}

// init this class
$baselineController = new bfBaseline($dataObj);

// Run the tool method
$baselineController->run();
