<?php
/*
 * @package   bfNetwork
 * @copyright Copyright (C) 2011,2012,2013,2014,2015,2016,2017,2018,2019,2020,2021,2022 Blue Flame Digital Solutions Ltd. All rights reserved.
 * @license   GNU General Public License version 3 or later
 *
 * @see       https://mySites.guru/
 * @see       https://www.phil-taylor.com/
 *
 * @author    Phil Taylor / Blue Flame Digital Solutions Limited.
 *
 * bfNetwork is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * bfNetwork is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this package.  If not, see http://www.gnu.org/licenses/
 *
 * If you have any questions regarding this code, please contact phil@phil-taylor.com
 */

/*
 * If we have not already included bfEncrypt then this must be a direct call, and
 * so we need to decrypt the incoming request.
 */
if (! class_exists('bfEncrypt')) {
    require 'bfEncrypt.php';
}

/*
 * Ok so this is stupid, but we are dealing with XML Parsing on crappy servers on sites with 100+ extensions installed - gulp!
 * 5 Mins! GULP - Most well configured servers will probably not honour this, but in our live tests on crappy servers this seems to work
 */
@set_time_limit(60 * 5);

/**
 * If we have got here then we have already passed through decrypting the encrypted header and so we are sure we are now
 * secure and no one else cannot run the code below.
 */
final class bfExtensions
{
    const PLUGIN = 'plugin';

    const THEME  = 'theme';

    /**
     * Incoming decrypted vars from the request.
     *
     * @var stdClass
     */
    private $_dataObj;

    /**
     * We pass the command to run as a simple integer in our encrypted request this is mainly to speed up the decryption
     * process, plus its a single digit(or 2) rather than a huge string to remember :-).
     */
    private $_methods = array(
        1 => 'doUpdate',
        2 => 'installExtensionFromUrl',
    );

    /**
     * I inject the request to the object.
     *
     * @param stdClass $dataObj
     */
    public function __construct($dataObj = null)
    {
        // init Joomla
        require 'bfInitWordpress.php';

        // Set the request vars
        $this->_dataObj = $dataObj;
    }

    /**
     * I'm the controller - I run methods based on the request integer.
     */
    public function run()
    {
        if (property_exists($this->_dataObj, 'c')) {
            $c = (int) $this->_dataObj->c;
            if (array_key_exists($c, $this->_methods)) {
                // call the right method
                $this->{$this->_methods[$c]} ();
            } else {
                // Die if an unknown function
                bfEncrypt::reply('error', 'No Such method #err1 - ' . $c);
            }
        } else {
            // Die if an unknown function
            bfEncrypt::reply('error', 'No Such method #err2');
        }
    }

    /**
     * Install a plugin from a given URL remotely.
     */
    public function installExtensionFromUrl()
    {
        require_once ABSPATH . 'wp-admin/includes/file.php';
        require_once ABSPATH . 'wp-admin/includes/plugin.php';
        include_once ABSPATH . 'wp-admin/includes/plugin-install.php';
        include_once ABSPATH . 'wp-admin/includes/class-wp-upgrader.php';

        $contents = file_get_contents($this->_dataObj->url);

        $fileName        = 'bf_remoteinstall_' . rand() . '.zip';
        $_GET['package'] = $fileName;

        $tmpFile = wp_upload_dir()['basedir'] . '/' . $fileName;
        file_put_contents($tmpFile, $contents);

        $file_upload = new File_Upload_Upgrader('pluginzip', 'package');
        $filename    = $file_upload->filename;

        $upgrader = new Plugin_Upgrader(new Automatic_Upgrader_Skin());
        $result   = $upgrader->install($file_upload->package);

        $destination_folder = $upgrader->result['destination_name'];
        $pluginDetails      = get_plugins('/' . $destination_folder);

        if (true === $result) {
            $activateresult = activate_plugin($destination_folder . '/' . array_keys($pluginDetails)[0]);
            if (null !== $activateresult) {
                $activateresult = new WP_Error('mywpguru', 'Installed but did not activate correctly, please check your site.');
            }
        } else {
            $activateresult = new WP_Error('mywpguru', 'Never installed correctly, Maybe its already installed?');
        }

        $file_upload->cleanup();
        @unlink($tmpFile);

        if (null !== $activateresult) {
            $result = false;
        }

        $res = array(
            'result'  => ! $result ? 'error' : 'success',
            'msgType' => ($result && null === $activateresult) ? 'success' : 'error',
            'message' => null             === $activateresult ? 'Plugin Installed And Activated OK - CHECK YOUR SITE!' : 'ERROR: ' . implode(
                ', ',
                $activateresult->get_error_messages()
            ),
        );

        bfEncrypt::reply($res['result'], $res);
    }

    /**
     * Updates a plugin.
     */
    public function doUpdate()
    {
        $pluginSlug = $this->_dataObj->wpplugin;

        require_once ABSPATH . 'wp-admin/includes/plugin-install.php';
        require_once ABSPATH . 'wp-admin/includes/theme-install.php';
        require_once ABSPATH . 'wp-admin/includes/update.php';
        require_once ABSPATH . 'wp-admin/includes/plugin.php';
        require_once ABSPATH . 'wp-admin/includes/theme.php';
        require_once ABSPATH . 'wp-admin/includes/file.php';
        include_once ABSPATH . 'wp-admin/includes/class-wp-upgrader.php';

        wp_clean_update_cache();
        wp_clean_plugins_cache();
        wp_clean_themes_cache();
        wp_update_plugins();
        wp_update_themes();

        $pluginToUpdate = null;
        if (preg_match('/THEME/', $pluginSlug)) {
            $type           = self::THEME;
            $pluginSlug     = str_replace('THEME/', '', $pluginSlug);
            $themes         = get_theme_updates();
            $pluginToUpdate = $themes[$pluginSlug];
        } else {
            $type    = self::PLUGIN;
            $plugins = get_plugin_updates();
            foreach ($plugins as $slug => $plugin) {
                if ($pluginToUpdate) {
                    continue;
                }

                if ($slug == $pluginSlug) {
                    $pluginToUpdateSlug = $slug;
                    $pluginToUpdate     = $plugin;
                    continue;
                }

                // PowerPack for Beaver Builder where the slug is not the same as the plugin slug
                // bpowerpack/bb-powerpack.php !== bb-powerpack/bb-powerpack.php
                if (property_exists($plugin, 'update')) {
                    if (property_exists($plugin->update, 'slug')) {
                        if ($pluginSlug == $plugin->update->slug . '/' . $plugin->update->slug . '.php') {
                            $pluginToUpdateSlug = $slug;
                            $pluginToUpdate     = $plugin;
                        }
                    }
                }
            }
        }

        if (! $pluginToUpdate) {
            $status                     = array();
            $status['RESULT']           = 'error';
            $status['errorCode']        = 'No slug match';
            $status['errorMessage']     = sprintf('No slug match, was looking for %s', $pluginSlug);
            $status['availablePlugins'] = $plugins;
            bfEncrypt::reply('error', $status);
        }

        $status = array(
            'update'     => $type,
            'slug'       => '',
            'oldVersion' => '',
            'newVersion' => '',
        );

        switch ($type) {
            case self::PLUGIN:
                $SLUG_TO_UPGRADE      = $pluginToUpdate->update->slug;
                $PLUGIN_TO_UPGRADE    = $pluginToUpdate->update->plugin ?: $pluginToUpdateSlug;
                $plugin               = plugin_basename(sanitize_text_field(wp_unslash($PLUGIN_TO_UPGRADE)));
                $plugin_data          = get_plugin_data(WP_PLUGIN_DIR . '/' . $plugin);

                $status['plugin']     = $plugin;
                $status['pluginName'] = $plugin_data['Name'];
                $status['slug']       = sanitize_key(wp_unslash($SLUG_TO_UPGRADE));

                if ($plugin_data['Version']) {
                    /* translators: %s: Plugin version */
                    $status['oldVersion'] = sprintf(__('Version %s'), $plugin_data['Version']);
                }

                // Wrap the upgrader in a skin for no output
                $skin     = new WP_Ajax_Upgrader_Skin();
                $upgrader = new Plugin_Upgrader($skin);
                // Do the upgrade
                $status['pluginToUpdate']     = $plugin;
                $result                       = $upgrader->bulk_upgrade(array($plugin));
                break;
            case self::THEME:
                $SLUG_TO_UPGRADE   = $pluginSlug;
                $status['slug']    = sanitize_key(wp_unslash($SLUG_TO_UPGRADE));
                $skin              = new WP_Ajax_Upgrader_Skin();

                $upgrader   = new Theme_Upgrader($skin);
                $stylesheet = $plugin= preg_replace('/[^A-z0-9_\-]/', '', wp_unslash($pluginSlug));

                $result   = $upgrader->bulk_upgrade(array($stylesheet));
                break;
        }

        // store an array of messages
        $status['debug']    = $skin->get_upgrade_messages();
        $status['feedback'] = $skin->feedback($result);

        if (is_wp_error($skin->result)) {
            $status['RESULT']       = 'error';
            $status['errorCode']    = $skin->result->get_error_code();
            $status['errorMessage'] = $skin->result->get_error_message();
        } elseif ($skin->get_errors()->get_error_code()) {
            $status['RESULT']       = 'error';
            $status['errorMessage'] = $skin->get_error_messages();
            $status['errorCode']    = $skin->get_errors()->get_error_code();
        } elseif (is_array($result) && ! empty($result[$plugin])) {
            // It worked - 🎉 Party!
            $status['RESULT'] = 'OK';

            $plugin_update_data = current($result);

            /*
             * If the `update_plugins` site transient is empty (e.g. when you update
             * two plugins in quick succession before the transient repopulates),
             * this may be the return.
             *
             * Preferably something can be done to ensure `update_plugins` isn't empty.
             * For now, surface some sort of error here.
             */
            if (true === $plugin_update_data) {
                $status['RESULT']       = 'error';
                $status['errorMessage'] = $type . ' update failed';
            }

            $plugin_data = get_plugins('/' . $result[$plugin]['destination_name']);
            $plugin_data = reset($plugin_data);

            if ($plugin_data['Version']) {
                /* translators: %s: Plugin version */
                $status['newVersion'] = sprintf(__('Version %s'), $plugin_data['Version']);
            }
        } elseif (false === $result) {
            global $wp_filesystem;

            $status['errorCode']    = 'unable_to_connect_to_filesystem';
            $status['errorMessage'] = __('Unable to connect to the filesystem. Please confirm your credentials.');
            $status['RESULT']       = 'error';

            // Pass through the error from WP_Filesystem if one was raised.
            if ($wp_filesystem instanceof WP_Filesystem_Base && is_wp_error(
                $wp_filesystem->errors
            ) && $wp_filesystem->errors->get_error_code()) {
                $status['errorMessage'] = esc_html($wp_filesystem->errors->get_error_message());
            }
        } else {
            // An unhandled error occurred.
            $status['errorMessage'] = $type . ' Plugin update failed, No idea why!.';
            $status['RESULT']       = 'error';
        }

        bfEncrypt::reply($status['RESULT'], $status);
    }
}
