<?php
/*
 * @package   bfNetwork
 * @copyright Copyright (C) 2011,2012,2013,2014,2015,2016,2017,2018,2019,2020,2021,2022 Blue Flame Digital Solutions Ltd. All rights reserved.
 * @license   GNU General Public License version 3 or later
 *
 * @see       https://mySites.guru/
 * @see       https://www.phil-taylor.com/
 *
 * @author    Phil Taylor / Blue Flame Digital Solutions Limited.
 *
 * bfNetwork is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * bfNetwork is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this package.  If not, see http://www.gnu.org/licenses/
 *
 * If you have any questions regarding this code, please contact phil@phil-taylor.com
 */

// No Direct Access - we are INSIDE Wordpress for this file to load, unlike much of the other files in this folder.
if (! defined('ABSPATH')) {
    exit;
} // Exit if accessed directly

if (! defined('BF_OK')) {
    define('BF_OK', 1);
}

// ensure our dependencies are loaded
require_once 'bfEvents.php';
require_once 'bfDb.php';
require_once 'bfPreferences.php';
require_once 'bfLog.php';
require_once 'bfActivitylog.php';

class PlgSystemBfnetwork
{
    public function __construct()
    {
        $this->addHooks();
        $this->loadMU();
    }

    /**
     * Add our hooks to the stack.
     */
    private function addHooks()
    {
        add_action('shutdown', array($this, 'mywpguru_shutdown_callback'), 10, 0);
        add_action('wp_login', array($this, 'onUserLogin'), 10, 2);
        add_action('wp_logout', array($this, 'onUserLogout'), 10, 0);
        add_action('user_register', array($this, 'onUserRegister'), 10, 1);
        add_action('show_user_profile', array($this, 'onUserEdit'), 10, 2);
        add_action('edit_user_profile', array($this, 'onUserEdit'), 10, 2);
        add_action('edit_user_profile_update', array($this, 'onUserSave'), 10, 2);
        //        add_action('edit_user_profile_update', array($this, 'onUserSave'), 10, 2);
        add_action('personal_options_update', array($this, 'onUserSave'), 10, 2);
    }

    private function loadMU()
    {
        $mu_options = array(
            'disable_xmlrpc.php',
            'disable_fileediting.php',
            'disable_emojis.php',
            'remove_jquery_migrate.php',
            'remove_admin_footer.php',
            'remove_generator_tag.php',
            'remove_shortlink.php',
            'remove_adminbar_wp.php',
            'disable_application_passwords.php',
            'disable_comment_links.php',
            'disable_howdy.php',
            'disable_nags.php',
            'disable_frontendmenubar.php',
        );
        foreach ($mu_options as $file) {
            if (file_exists(dirname(__FILE__) . '/mu_options/enabled/' . $file)) {
                require_once dirname(__FILE__) . '/mu_options/enabled/' . $file;
            }
        }
    }

    /**
     * When a users profile is saved in admin.
     *
     * @param $userId
     */
    public function onUserSave($userId)
    {
        bfLog::log(__METHOD__);

        $user = get_user_by('id', $userId);

        if (! $userId || ! $user) {
            return;
        }

        $loggedInUser = wp_get_current_user();

        bfActivitylog::getInstance()->log(
            $loggedInUser->user_login,
            $loggedInUser->ID,
            sprintf(
                '%s:%s saved user profile for %s:%s',
                $loggedInUser->ID,
                $loggedInUser->user_login,
                $user->ID,
                $user->user_login
            ),//what
            'bfnetwork/bfTools.php', //where
            bfEvents::onUserModified,// where_id
            null,// ip
            null,// useragent
            json_encode(array(
                'viewed_by_id'   => $loggedInUser->ID,
                'viewed_by_name' => $loggedInUser->user_login,
                'id'             => $user->ID,
                'username'       => $user->user_login,
            )),// meta
            '',// action
            bfEvents::onUserModified,//alertname
            bfEvents::onUserModified //const
        );
    }

    /**
     * When a users profile is viewed in admin.
     *
     * @param $user
     */
    public function onUserEdit($user)
    {
        bfLog::log(__METHOD__);

        $loggedInUser = wp_get_current_user();

        bfActivitylog::getInstance()->log(
            $loggedInUser->user_login,
            $loggedInUser->ID,
            'viewed user profile',
            'onUserEdit',
            '0',
            null,
            null,
            json_encode(array(
                'viewed_by_id'   => $loggedInUser->ID,
                'viewed_by_name' => $loggedInUser->user_login,
                'id'             => $user->ID,
                'username'       => $user->user_login,
            )),
            'com_users',
            'alerting_viewuser',
            bfEvents::onUserViewed
        );
    }

    /**
     * Fires AFTER a successful registration.
     */
    public function onUserRegister($userId)
    {
        bfLog::log(__METHOD__);

        $user = get_user_by('id', $userId);

        if (! $userId || ! $user) {
            return;
        }

        $loggedInUser = wp_get_current_user();

        bfActivitylog::getInstance()->log(
            $user->user_login,
            $user->ID,
            'created a new user',
            0 !== $loggedInUser->ID ? 'onUserAddedByAdmin' : 'onUserRegister',
            '0',
            null,
            null,
            json_encode(array(
                'created_by_id'   => $loggedInUser->ID,
                'created_by_name' => $loggedInUser->user_login,
                'id'              => $user->ID,
                'username'        => $user->user_login,
                'roles'           => $user->roles,
            )),
            'com_users',
            'alerting_newuser',
            bfEvents::onUserCreated
        );
    }

    /**
     * Fires AFTER a successful logout.
     */
    public function onUserLogout()
    {
        bfLog::log(__METHOD__);

        $user = wp_get_current_user();

        // Only log administrators
        if (! in_array('administrator', $user->roles, true)) {
            bfLog::log('Not an admin');

            $constKey = bfEvents::onUserLogout;
            $alerting = 'alerting_userlogout';
        } else {
            $constKey = bfEvents::onAdminLogout;
            $alerting = 'alerting_superadminlogout';
        }

        bfActivitylog::getInstance()->log(
            $user->user_login,
            $user->ID,
            $user->ID . ':' . $user->user_login . ' logged out',
            'onUserLogout',
            '0',
            null,
            null,
            json_encode(array(
                'ID'          => $user->ID,
                'username'    => $user->user_login,
                'redirect_to' => @$_REQUEST['redirect_to'],
            )),
            'logout',
            $alerting,
            $constKey
        );
    }

    /**
     * Fires AFTER a successful login.
     */
    public function onUserLogin($user_login, $user)
    {
        bfLog::log(__METHOD__);

        // Only log administrators
        if (! in_array('administrator', $user->roles, true)) {
            $constKey = bfEvents::onUserLogin;
            $alerting = 'alerting_userlogin';
        } else {
            $constKey = bfEvents::onAdminLogin;
            $alerting = 'alerting_superadminlogin';
        }

        bfActivitylog::getInstance()->log(
            $user->user_login,
            $user->ID,
            $user->ID . ':' . $user->user_login . ' logged in',
            'onUserLogin',
            '0',
            null,
            null,
            json_encode(array(
                'ID'          => $user->ID,
                'username'    => $user->user_login,
                'redirect_to' => @$_REQUEST['redirect_to'],
            )),
            'login',
            $alerting,
            $constKey
        );
    }

    /**
     * check for file modifications.
     */
    public function mywpguru_shutdown_callback()
    {
//        echo '<pre>';
//        print_r( $GLOBALS['wp_actions'] );
//        echo '</pre>';

        bfLog::log(__METHOD__ . ' running');

        $prefs = new bfPreferences();
        $prefs->getPreferences(); // force creation of prefs file if needed

        $prefs       = new bfPreferences();
        $preferences = $prefs->getPreferences();

        if (property_exists($preferences, 'alerting_filewatchlist')) {
            $fileList = json_decode($preferences->alerting_filewatchlist);
        } else {
            $fileList = json_decode(json_encode($prefs->default_alerting_filewatchlist));
        }

        foreach ($fileList as $file) {
            if (! file_exists(ABSPATH . $file)) {
                continue;
            }

            $createLock = false;

            $pathinfo = pathinfo($file);

            $md5LockFile = str_replace('//', '/', ABSPATH . $pathinfo['dirname'] . '/.mywpguru.' . basename($file) . '.md5');

            bfLog::log('LOCK FILE = ' . $md5LockFile);

            $currentMd5 = md5_file(ABSPATH . $file);
            bfLog::log('CURRENT MD5 = ' . $currentMd5);

            if (file_exists($md5LockFile)) {
                bfLog::log('LOCK FILE EXISTS = ' . $md5LockFile);
                $lastMd5 = file_get_contents($md5LockFile);
            } else {
                bfLog::log('LOCK FILE NOT EXISTS = ' . $md5LockFile);
                $lastMd5 = md5_file(ABSPATH . $file);

                bfLog::log("CREATING LOCK FILE with $currentMd5");
                // @ as not to upset crap servers :-(
                $res = @file_put_contents($md5LockFile, $currentMd5);
                bfLog::log('file_put_contents was = ' . $res);

                // if we could not write the lock file then bail!
                if (! file_exists($md5LockFile)) {
                    return;
                }
            }

            bfLog::log("COMPARING =   $lastMd5 !!! $currentMd5");
            if ($lastMd5 !== $currentMd5) {
                $createLock = true;
                bfLog::log("ALERTING COMPARING !==  >   $lastMd5 !!! $currentMd5");
                bfActivitylog::getInstance()->log(
                    'Someone',
                    '-911',
                    'modified file',
                    $file,
                    null,
                    'system',
                    null,
                    null,
                    null,
                    'alerting_filewatchlist_alert',
                    bfEvents::onFileModified
                );
            }

            if (true === $createLock) {
                bfLog::log("CREATING LOCK FILE with $currentMd5");
                // @ as not to upset crap servers :-(
                $res = @file_put_contents($md5LockFile, $currentMd5);
                bfLog::log('file_put_contents was = ' . $res);
            }
        }
        bfLog::log(__METHOD__ . ' finished');
    }
}

new PlgSystemBfnetwork();
