<?php
/*
 * @package   bfNetwork
 * @copyright Copyright (C) 2011,2012,2013,2014,2015,2016,2017,2018,2019,2020,2021,2022 Blue Flame Digital Solutions Ltd. All rights reserved.
 * @license   GNU General Public License version 3 or later
 *
 * @see       https://mySites.guru/
 * @see       https://www.phil-taylor.com/
 *
 * @author    Phil Taylor / Blue Flame Digital Solutions Limited.
 *
 * bfNetwork is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * bfNetwork is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this package.  If not, see http://www.gnu.org/licenses/
 *
 * If you have any questions regarding this code, please contact phil@phil-taylor.com
 */

namespace Akeeba\AdminTools\Library\Encrypt;

\defined('ADMINTOOLSINC') or exit();

/**
 * Generates cryptographically-secure random values.
 */
class Randval
{
    /**
     * Returns a cryptographically secure random value.
     *
     * @param int $bytes How many bytes to return
     *
     * @return string
     */
    public function generate($bytes = 32)
    {
        if (\extension_loaded('openssl') && (version_compare(PHP_VERSION, '5.3.4') >= 0 || IS_WIN)) {
            $strong    = false;
            $randBytes = openssl_random_pseudo_bytes($bytes, $strong);

            if ($strong) {
                return $randBytes;
            }
        }

        if (\extension_loaded('mcrypt')) {
            return mcrypt_create_iv($bytes, MCRYPT_DEV_URANDOM);
        }

        return $this->genRandomBytes($bytes);
    }

    /**
     * Generate random bytes. Adapted from Joomla! 3.2.
     *
     * @param int $length Length of the random data to generate
     *
     * @return string Random binary data
     */
    public function genRandomBytes($length = 32)
    {
        $length = (int) $length;
        $sslStr = '';

        /*
         * Collect any entropy available in the system along with a number
         * of time measurements of operating system randomness.
         */
        $bitsPerRound  = 2;
        $maxTimeMicro  = 400;
        $shaHashLength = 20;
        $randomStr     = '';
        $total         = $length;

        // Check if we can use /dev/urandom.
        $urandom = false;
        $handle  = null;

        // This is PHP 5.3.3 and up
        if (\function_exists('stream_set_read_buffer') && @is_readable('/dev/urandom')) {
            $handle = @fopen('/dev/urandom', 'r');

            if ($handle) {
                $urandom = true;
            }
        }

        while ($length > \strlen($randomStr)) {
            $bytes = ($total > $shaHashLength) ? $shaHashLength : $total;
            $total -= $bytes;

            /*
             * Collect any entropy available from the PHP system and filesystem.
             * If we have ssl data that isn't strong, we use it once.
             */
            $entropy = rand() . uniqid(mt_rand(), true) . $sslStr;
            $entropy .= implode('', @fstat(fopen(__FILE__, 'r')));
            $entropy .= memory_get_usage();
            $sslStr = '';

            if ($urandom) {
                stream_set_read_buffer($handle, 0);
                $entropy .= @fread($handle, $bytes);
            } else {
                /*
                 * There is no external source of entropy so we repeat calls
                 * to mt_rand until we are assured there's real randomness in
                 * the result.
                 *
                 * Measure the time that the operations will take on average.
                 */
                $samples  = 3;
                $duration = 0;

                for ($pass = 0; $pass < $samples; ++$pass) {
                    $microStart = microtime(true) * 1000000;
                    $hash       = sha1(mt_rand(), true);

                    for ($count = 0; $count < 50; ++$count) {
                        $hash = sha1($hash, true);
                    }

                    $microEnd = microtime(true) * 1000000;
                    $entropy .= $microStart . $microEnd;

                    if ($microStart >= $microEnd) {
                        $microEnd += 1000000;
                    }

                    $duration += $microEnd - $microStart;
                }

                $duration = $duration / $samples;

                /*
                 * Based on the average time, determine the total rounds so that
                 * the total running time is bounded to a reasonable number.
                 */
                $rounds = (int) (($maxTimeMicro / $duration) * 50);

                /*
                 * Take additional measurements. On average we can expect
                 * at least $bitsPerRound bits of entropy from each measurement.
                 */
                $iter = $bytes * (int) ceil(8 / $bitsPerRound);

                for ($pass = 0; $pass < $iter; ++$pass) {
                    $microStart = microtime(true);
                    $hash       = sha1(mt_rand(), true);

                    for ($count = 0; $count < $rounds; ++$count) {
                        $hash = sha1($hash, true);
                    }

                    $entropy .= $microStart . microtime(true);
                }
            }

            $randomStr .= sha1($entropy, true);
        }

        if ($urandom) {
            @fclose($handle);
        }

        return substr($randomStr, 0, $length);
    }

    /**
     * Generates a random string with the specified length. WARNING: You get to specify the number of random characters
     * in the string, not the number of random bytes. The character pool is 64 characters (6 bits) long. The entropy of
     * your string is 6 * $characters bits. This means that a random string of 32 characters has an entropy of 192 bits
     * whereas a random sequence of 32 bytes returned by generate() has an entropy of 8 * 32 = 256 bits.
     *
     * @param int $characters
     *
     * @return string
     */
    public function generateString($characters = 32)
    {
        $sourceString = str_split('abcdefghijklmnopqrstuvwxyz-ABCDEFGHIJKLMNOPQRSTUVWXYZ_0123456789', 1);
        $ret          = '';

        $bytes     = ceil($characters / 4) * 3;
        $randBytes = $this->generate($bytes);

        for ($i = 0; $i < $bytes; $i += 3) {
            $subBytes = substr($randBytes, $i, 3);
            $subBytes = str_split($subBytes, 1);
            $subBytes = \ord($subBytes[0]) * 65536 + \ord($subBytes[1]) * 256 + \ord($subBytes[2]);
            $subBytes = $subBytes & bindec('00000000111111111111111111111111');

            $b    = array();
            $b[0] = $subBytes  >> 18;
            $b[1] = ($subBytes >> 12) & bindec('111111');
            $b[2] = ($subBytes >> 6)  & bindec('111111');
            $b[3] = $subBytes         & bindec('111111');

            $ret .= $sourceString[$b[0]] . $sourceString[$b[1]] . $sourceString[$b[2]] . $sourceString[$b[3]];
        }

        return substr($ret, 0, $characters);
    }
}
